<!DOCTYPE html>
<html lang="tr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Premium HLS Video Player</title>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
        }

        body {
            background: #0f0f1e;
            color: #fff;
            overflow: hidden;
            height: 100vh;
        }

        .video-container {
            position: relative;
            width: 100%;
            height: 100vh;
            overflow: hidden;
            background: #000;
        }

        #videoPlayer {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }

        .poster-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.9);
            background-size: cover;
            background-position: center;
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            z-index: 10;
            transition: opacity 0.8s ease;
        }

        .poster-overlay.hidden {
            opacity: 0;
            pointer-events: none;
        }

        .poster-title {
            font-size: 3.5rem;
            font-weight: 700;
            margin-bottom: 1.5rem;
            text-align: center;
            background: linear-gradient(90deg, #ff0066, #6600ff);
            -webkit-background-clip: text;
            background-clip: text;
            color: transparent;
            text-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
        }

        .poster-subtitle {
            font-size: 1.2rem;
            max-width: 600px;
            text-align: center;
            margin-bottom: 2.5rem;
            color: #aaa;
            line-height: 1.6;
        }

        .play-button {
            background: linear-gradient(90deg, #ff0066, #6600ff);
            border: none;
            color: white;
            padding: 18px 45px;
            font-size: 1.3rem;
            font-weight: 600;
            border-radius: 50px;
            cursor: pointer;
            display: flex;
            align-items: center;
            gap: 12px;
            transition: all 0.3s ease;
            box-shadow: 0 10px 25px rgba(255, 0, 102, 0.3);
        }

        .play-button:hover {
            transform: scale(1.05);
            box-shadow: 0 15px 30px rgba(255, 0, 102, 0.4);
        }

        .controls-overlay {
            position: absolute;
            bottom: 0;
            left: 0;
            width: 100%;
            background: linear-gradient(to top, rgba(0, 0, 0, 0.9), transparent);
            padding: 20px 30px;
            display: flex;
            flex-direction: column;
            gap: 15px;
            opacity: 0;
            transition: opacity 0.3s ease;
            z-index: 5;
        }

        .video-container:hover .controls-overlay {
            opacity: 1;
        }

        .progress-bar-container {
            width: 100%;
            height: 8px;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 4px;
            overflow: hidden;
            cursor: pointer;
        }

        .progress-bar {
            height: 100%;
            background: linear-gradient(90deg, #ff0066, #6600ff);
            width: 0%;
            border-radius: 4px;
            transition: width 0.1s;
        }

        .controls-bottom {
            display: flex;
            justify-content: space-between;
            align-items: center;
        }

        .left-controls, .right-controls {
            display: flex;
            align-items: center;
            gap: 20px;
        }

        .control-btn {
            background: transparent;
            border: none;
            color: white;
            font-size: 1.5rem;
            cursor: pointer;
            transition: all 0.2s ease;
        }

        .control-btn:hover {
            color: #ff0066;
            transform: scale(1.1);
        }

        .volume-container {
            display: flex;
            align-items: center;
            gap: 10px;
        }

        .volume-slider {
            width: 80px;
            height: 5px;
            -webkit-appearance: none;
            background: rgba(255, 255, 255, 0.2);
            border-radius: 5px;
            outline: none;
        }

        .volume-slider::-webkit-slider-thumb {
            -webkit-appearance: none;
            width: 16px;
            height: 16px;
            border-radius: 50%;
            background: #ff0066;
            cursor: pointer;
        }

        .time-display {
            font-size: 1rem;
            font-weight: 500;
            color: #ddd;
            min-width: 110px;
        }

        .fullscreen-btn {
            margin-left: 10px;
        }

        .logo-container {
            position: absolute;
            bottom: 30px;
            right: 30px;
            z-index: 6;
            opacity: 0.8;
            transition: opacity 0.3s ease;
        }

        .logo-container:hover {
            opacity: 1;
        }

        .logo {
            width: 80px;
            height: auto;
            filter: drop-shadow(0 0 10px rgba(255, 255, 255, 0.2));
        }

        .quality-selector {
            position: relative;
        }

        .quality-btn {
            background: rgba(0, 0, 0, 0.7);
            color: white;
            border: 1px solid rgba(255, 255, 255, 0.2);
            padding: 8px 15px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.9rem;
        }

        .quality-menu {
            position: absolute;
            bottom: 40px;
            background: rgba(0, 0, 0, 0.9);
            border: 1px solid rgba(255, 255, 255, 0.1);
            border-radius: 5px;
            padding: 10px 0;
            display: none;
            min-width: 120px;
        }

        .quality-menu.show {
            display: block;
        }

        .quality-option {
            padding: 10px 20px;
            color: white;
            cursor: pointer;
            transition: background 0.2s;
        }

        .quality-option:hover {
            background: rgba(255, 255, 255, 0.1);
        }

        .loading-spinner {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            width: 70px;
            height: 70px;
            border: 5px solid rgba(255, 255, 255, 0.1);
            border-top: 5px solid #ff0066;
            border-radius: 50%;
            animation: spin 1s linear infinite;
            z-index: 4;
            display: none;
        }

        @keyframes spin {
            0% { transform: translate(-50%, -50%) rotate(0deg); }
            100% { transform: translate(-50%, -50%) rotate(360deg); }
        }

        .notification {
            position: absolute;
            top: 30px;
            left: 50%;
            transform: translateX(-50%);
            background: rgba(0, 0, 0, 0.8);
            padding: 15px 30px;
            border-radius: 8px;
            border-left: 5px solid #ff0066;
            display: none;
            z-index: 7;
            box-shadow: 0 10px 25px rgba(0, 0, 0, 0.3);
        }

        .error-message {
            position: absolute;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.9);
            padding: 30px;
            border-radius: 15px;
            text-align: center;
            max-width: 500px;
            width: 90%;
            z-index: 8;
            display: none;
            border: 2px solid #ff0066;
        }

        .error-title {
            color: #ff0066;
            font-size: 1.8rem;
            margin-bottom: 15px;
        }

        .error-text {
            color: #ddd;
            margin-bottom: 20px;
            line-height: 1.5;
        }

        .source-buttons {
            display: flex;
            gap: 15px;
            justify-content: center;
            flex-wrap: wrap;
        }

        .source-btn {
            background: rgba(255, 255, 255, 0.1);
            border: none;
            color: white;
            padding: 12px 20px;
            border-radius: 8px;
            cursor: pointer;
            transition: all 0.3s ease;
        }

        .source-btn:hover {
            background: rgba(255, 0, 102, 0.2);
            transform: translateY(-2px);
        }

        .source-btn.active {
            background: linear-gradient(90deg, #ff0066, #6600ff);
        }

        @media (max-width: 768px) {
            .poster-title {
                font-size: 2.5rem;
            }
            
            .controls-overlay {
                padding: 15px;
            }
            
            .control-btn {
                font-size: 1.3rem;
            }
            
            .logo-container {
                bottom: 20px;
                right: 20px;
            }
            
            .logo {
                width: 60px;
            }
            
            .source-buttons {
                flex-direction: column;
            }
        }

        @media (max-width: 480px) {
            .poster-title {
                font-size: 2rem;
            }
            
            .poster-subtitle {
                font-size: 1rem;
                padding: 0 15px;
            }
            
            .play-button {
                padding: 15px 30px;
                font-size: 1.1rem;
            }
            
            .controls-bottom {
                flex-wrap: wrap;
                justify-content: center;
                gap: 15px;
            }
            
            .logo-container {
                position: relative;
                bottom: auto;
                right: auto;
                margin-top: 20px;
                text-align: center;
                width: 100%;
            }
        }
    </style>
</head>
<body>
    <div class="video-container">
        <!-- Video Poster Overlay -->
        <div class="poster-overlay" id="posterOverlay">
            <h1 class="poster-title">Premium HLS Video Player</h1>
            <p class="poster-subtitle">Gelişmiş HLS video oynatıcı ile yüksek kaliteli video deneyimi. Aşağıdan video kaynağı seçebilirsiniz.</p>
            
            <div class="source-buttons">
                <button class="source-btn active" data-source="original">
                    <i class="fas fa-link"></i> Orjinal Kaynak
                </button>
                <button class="source-btn" data-source="test1">
                    <i class="fas fa-video"></i> Test Videosu 1
                </button>
                <button class="source-btn" data-source="test2">
                    <i class="fas fa-film"></i> Test Videosu 2
                </button>
            </div>
            
            <button class="play-button" id="playButton">
                <i class="fas fa-play"></i>
                Videoyu Oynat
            </button>
        </div>

        <!-- Video Player -->
        <video id="videoPlayer" playsinline></video>

        <!-- Loading Spinner -->
        <div class="loading-spinner" id="loadingSpinner"></div>

        <!-- Error Message -->
        <div class="error-message" id="errorMessage">
            <h2 class="error-title"><i class="fas fa-exclamation-triangle"></i> Video Yüklenemedi</h2>
            <p class="error-text" id="errorText">Video kaynağına bağlanılamıyor. Bu durum CORS politikalarından veya geçersiz bir video linkinden kaynaklanıyor olabilir.</p>
            <p class="error-text">Lütfen alternatif bir video kaynağı seçin veya video linkinizi kontrol edin.</p>
            <button class="play-button" id="retryButton" style="margin-top: 20px;">
                <i class="fas fa-redo"></i>
                Tekrar Dene
            </button>
        </div>

        <!-- Controls Overlay -->
        <div class="controls-overlay">
            <div class="progress-bar-container" id="progressContainer">
                <div class="progress-bar" id="progressBar"></div>
            </div>
            
            <div class="controls-bottom">
                <div class="left-controls">
                    <button class="control-btn" id="playPauseBtn">
                        <i class="fas fa-play"></i>
                    </button>
                    
                    <div class="volume-container">
                        <button class="control-btn" id="volumeBtn">
                            <i class="fas fa-volume-up"></i>
                        </button>
                        <input type="range" class="volume-slider" id="volumeSlider" min="0" max="1" step="0.01" value="1">
                    </div>
                    
                    <div class="time-display">
                        <span id="currentTime">00:00</span> / <span id="duration">00:00</span>
                    </div>
                </div>
                
                <div class="right-controls">
                    <div class="quality-selector">
                        <button class="quality-btn" id="qualityBtn">Kalite: Otomatik</button>
                        <div class="quality-menu" id="qualityMenu">
                            <div class="quality-option" data-quality="auto">Otomatik</div>
                            <div class="quality-option" data-quality="1080">1080p</div>
                            <div class="quality-option" data-quality="720">720p</div>
                            <div class="quality-option" data-quality="480">480p</div>
                        </div>
                    </div>
                    
                    <button class="control-btn fullscreen-btn" id="fullscreenBtn">
                        <i class="fas fa-expand"></i>
                    </button>
                </div>
            </div>
        </div>

        <!-- Logo Container -->
        <div class="logo-container">
            <img src="/uploads/logo/original/logo-5025.webp" alt="Premium Logo" class="logo">
        </div>

        <!-- Notification -->
        <div class="notification" id="notification">
            <span id="notificationText">Video oynatılmaya hazır</span>
        </div>
    </div>

    <!-- HLS.js Library -->
    <script src="https://cdn.jsdelivr.net/npm/hls.js@latest"></script>
    
    <script>
        // DOM Elements
        const videoPlayer = document.getElementById('videoPlayer');
        const posterOverlay = document.getElementById('posterOverlay');
        const playButton = document.getElementById('playButton');
        const playPauseBtn = document.getElementById('playPauseBtn');
        const progressBar = document.getElementById('progressBar');
        const progressContainer = document.getElementById('progressContainer');
        const volumeBtn = document.getElementById('volumeBtn');
        const volumeSlider = document.getElementById('volumeSlider');
        const currentTimeEl = document.getElementById('currentTime');
        const durationEl = document.getElementById('duration');
        const fullscreenBtn = document.getElementById('fullscreenBtn');
        const qualityBtn = document.getElementById('qualityBtn');
        const qualityMenu = document.getElementById('qualityMenu');
        const loadingSpinner = document.getElementById('loadingSpinner');
        const notification = document.getElementById('notification');
        const notificationText = document.getElementById('notificationText');
        const errorMessage = document.getElementById('errorMessage');
        const errorText = document.getElementById('errorText');
        const retryButton = document.getElementById('retryButton');
        const sourceButtons = document.querySelectorAll('.source-btn');

        // Video sources
        const videoSources = {
            original: 'https://hlssssss.oetaylandgidiyom.workers.dev/https://corestream.ronaldovurdu.help//hls/bein-sports-1.m3u8',
            test1: 'https://test-streams.mux.dev/x36xhzz/x36xhzz.m3u8',
            test2: 'https://content.jwplatform.com/manifests/vM7nH0Kl.m3u8'
        };

        // HLS Player instance
        let hls;
        let currentSource = 'original';
        
        // Initialize player
        function initPlayer(sourceUrl, sourceName) {
            // Reset state
            if (hls) {
                hls.destroy();
                hls = null;
            }
            
            // Show loading spinner
            loadingSpinner.style.display = 'block';
            errorMessage.style.display = 'none';
            
            // Update current source
            currentSource = sourceName;
            
            // Check if browser supports HLS natively
            if (videoPlayer.canPlayType('application/vnd.apple.mpegurl')) {
                // For Safari and other browsers with native HLS support
                videoPlayer.src = sourceUrl;
                
                videoPlayer.addEventListener('loadedmetadata', function() {
                    loadingSpinner.style.display = 'none';
                    showNotification(`${sourceName === 'original' ? 'Orjinal' : 'Test'} video yüklendi`);
                    initializeControls();
                });
                
                videoPlayer.addEventListener('error', function() {
                    handleVideoError();
                });
            } else if (Hls.isSupported()) {
                // For browsers that don't support HLS natively
                hls = new Hls({
                    debug: false,
                    enableWorker: true,
                    lowLatencyMode: true,
                    backBufferLength: 90
                });
                
                hls.loadSource(sourceUrl);
                hls.attachMedia(videoPlayer);
                
                hls.on(Hls.Events.MANIFEST_PARSED, function() {
                    loadingSpinner.style.display = 'none';
                    showNotification(`${sourceName === 'original' ? 'Orjinal' : 'Test'} video yüklendi`);
                    initializeControls();
                });
                
                hls.on(Hls.Events.ERROR, function(event, data) {
                    console.error('HLS error:', data);
                    handleVideoError(data.details || 'Bilinmeyen HLS hatası');
                });
            } else {
                loadingSpinner.style.display = 'none';
                showNotification('Tarayıcınız HLS video formatını desteklemiyor.');
            }
        }

        // Handle video error
        function handleVideoError(errorDetails = '') {
            loadingSpinner.style.display = 'none';
            
            let errorMsg = 'Video kaynağına bağlanılamıyor. ';
            
            if (currentSource === 'original') {
                errorMsg += 'Orjinal video linki çalışmıyor olabilir. Lütfen alternatif bir video kaynağı seçin.';
            } else {
                errorMsg += 'Test videosu yüklenemedi.';
            }
            
            if (errorDetails) {
                errorMsg += ` (Hata: ${errorDetails})`;
            }
            
            errorText.textContent = errorMsg;
            errorMessage.style.display = 'block';
            posterOverlay.classList.remove('hidden');
        }

        // Initialize controls
        function initializeControls() {
            // Update video duration
            const duration = videoPlayer.duration;
            durationEl.textContent = formatTime(duration);
            
            // Event listeners
            videoPlayer.addEventListener('timeupdate', updateProgress);
            videoPlayer.addEventListener('loadedmetadata', updateDuration);
            videoPlayer.addEventListener('waiting', showLoading);
            videoPlayer.addEventListener('playing', hideLoading);
            
            // Play button click event
            playButton.addEventListener('click', startPlayback);
            playPauseBtn.addEventListener('click', togglePlayPause);
            
            // Progress bar click event
            progressContainer.addEventListener('click', setProgress);
            
            // Volume controls
            volumeBtn.addEventListener('click', toggleMute);
            volumeSlider.addEventListener('input', updateVolume);
            
            // Fullscreen
            fullscreenBtn.addEventListener('click', toggleFullscreen);
            
            // Quality selector
            qualityBtn.addEventListener('click', toggleQualityMenu);
            document.querySelectorAll('.quality-option').forEach(option => {
                option.addEventListener('click', selectQuality);
            });
            
            // Keyboard shortcuts
            document.addEventListener('keydown', handleKeyPress);
            
            // Hide controls after 3 seconds of inactivity
            let controlsTimeout;
            videoPlayer.addEventListener('mousemove', () => {
                clearTimeout(controlsTimeout);
                document.querySelector('.controls-overlay').style.opacity = '1';
                controlsTimeout = setTimeout(() => {
                    document.querySelector('.controls-overlay').style.opacity = '0';
                }, 3000);
            });
        }

        // Start video playback
        function startPlayback() {
            posterOverlay.classList.add('hidden');
            errorMessage.style.display = 'none';
            
            // Check if video is ready
            if (videoPlayer.readyState < 2) {
                showNotification('Video henüz yüklenmedi, lütfen bekleyin...');
                return;
            }
            
            videoPlayer.play().then(() => {
                playPauseBtn.innerHTML = '<i class="fas fa-pause"></i>';
                showNotification('Video oynatılıyor...');
            }).catch(error => {
                showNotification('Video oynatılamadı: ' + error.message);
            });
        }

        // Toggle play/pause
        function togglePlayPause() {
            if (videoPlayer.paused) {
                videoPlayer.play().then(() => {
                    playPauseBtn.innerHTML = '<i class="fas fa-pause"></i>';
                    posterOverlay.classList.add('hidden');
                }).catch(error => {
                    showNotification('Video oynatılamadı: ' + error.message);
                });
            } else {
                videoPlayer.pause();
                playPauseBtn.innerHTML = '<i class="fas fa-play"></i>';
            }
        }

        // Update progress bar
        function updateProgress() {
            const currentTime = videoPlayer.currentTime;
            const duration = videoPlayer.duration;
            
            if (duration && !isNaN(duration)) {
                const progressPercent = (currentTime / duration) * 100;
                progressBar.style.width = `${progressPercent}%`;
                currentTimeEl.textContent = formatTime(currentTime);
            }
        }

        // Set progress on click
        function setProgress(e) {
            const width = this.clientWidth;
            const clickX = e.offsetX;
            const duration = videoPlayer.duration;
            
            if (duration && !isNaN(duration)) {
                videoPlayer.currentTime = (clickX / width) * duration;
            }
        }

        // Update video duration
        function updateDuration() {
            const duration = videoPlayer.duration;
            if (duration && !isNaN(duration)) {
                durationEl.textContent = formatTime(duration);
            }
        }

        // Format time (seconds to MM:SS)
        function formatTime(seconds) {
            if (isNaN(seconds) || seconds === Infinity) return '00:00';
            
            const mins = Math.floor(seconds / 60);
            const secs = Math.floor(seconds % 60);
            
            return `${mins.toString().padStart(2, '0')}:${secs.toString().padStart(2, '0')}`;
        }

        // Toggle mute
        function toggleMute() {
            videoPlayer.muted = !videoPlayer.muted;
            
            if (videoPlayer.muted) {
                volumeBtn.innerHTML = '<i class="fas fa-volume-mute"></i>';
                volumeSlider.value = 0;
            } else {
                volumeBtn.innerHTML = '<i class="fas fa-volume-up"></i>';
                volumeSlider.value = videoPlayer.volume;
            }
        }

        // Update volume
        function updateVolume() {
            const volume = volumeSlider.value;
            videoPlayer.volume = volume;
            
            if (volume == 0) {
                volumeBtn.innerHTML = '<i class="fas fa-volume-mute"></i>';
            } else if (volume < 0.5) {
                volumeBtn.innerHTML = '<i class="fas fa-volume-down"></i>';
            } else {
                volumeBtn.innerHTML = '<i class="fas fa-volume-up"></i>';
            }
            
            videoPlayer.muted = false;
        }

        // Toggle fullscreen
        function toggleFullscreen() {
            if (!document.fullscreenElement) {
                document.querySelector('.video-container').requestFullscreen().catch(err => {
                    console.error(`Fullscreen error: ${err.message}`);
                });
                fullscreenBtn.innerHTML = '<i class="fas fa-compress"></i>';
            } else {
                document.exitFullscreen();
                fullscreenBtn.innerHTML = '<i class="fas fa-expand"></i>';
            }
        }

        // Toggle quality menu
        function toggleQualityMenu() {
            qualityMenu.classList.toggle('show');
        }

        // Select quality
        function selectQuality(e) {
            const quality = e.target.getAttribute('data-quality');
            qualityBtn.textContent = `Kalite: ${quality === 'auto' ? 'Otomatik' : quality + 'p'}`;
            qualityMenu.classList.remove('show');
            
            // In a real implementation, you would switch to the selected quality level
            showNotification(`Kalite ${quality === 'auto' ? 'otomatik' : quality + 'p'} olarak ayarlandı`);
        }

        // Show loading spinner
        function showLoading() {
            loadingSpinner.style.display = 'block';
        }

        // Hide loading spinner
        function hideLoading() {
            loadingSpinner.style.display = 'none';
        }

        // Show notification
        function showNotification(message) {
            notificationText.textContent = message;
            notification.style.display = 'block';
            
            setTimeout(() => {
                notification.style.display = 'none';
            }, 3000);
        }

        // Handle keyboard shortcuts
        function handleKeyPress(e) {
            switch(e.key.toLowerCase()) {
                case ' ':
                case 'k':
                    e.preventDefault();
                    togglePlayPause();
                    break;
                case 'm':
                    e.preventDefault();
                    toggleMute();
                    break;
                case 'f':
                    e.preventDefault();
                    toggleFullscreen();
                    break;
                case 'arrowleft':
                    e.preventDefault();
                    videoPlayer.currentTime -= 10;
                    break;
                case 'arrowright':
                    e.preventDefault();
                    videoPlayer.currentTime += 10;
                    break;
                case 'arrowup':
                    e.preventDefault();
                    videoPlayer.volume = Math.min(1, videoPlayer.volume + 0.1);
                    volumeSlider.value = videoPlayer.volume;
                    updateVolume();
                    break;
                case 'arrowdown':
                    e.preventDefault();
                    videoPlayer.volume = Math.max(0, videoPlayer.volume - 0.1);
                    volumeSlider.value = videoPlayer.volume;
                    updateVolume();
                    break;
            }
        }

        // Source selection
        sourceButtons.forEach(button => {
            button.addEventListener('click', function() {
                const source = this.getAttribute('data-source');
                
                // Update active button
                sourceButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                // Show loading
                loadingSpinner.style.display = 'block';
                posterOverlay.classList.add('hidden');
                
                // Load new source
                initPlayer(videoSources[source], source);
            });
        });

        // Retry button
        retryButton.addEventListener('click', function() {
            initPlayer(videoSources[currentSource], currentSource);
        });

        // Close quality menu when clicking outside
        document.addEventListener('click', (e) => {
            if (!qualityBtn.contains(e.target) && !qualityMenu.contains(e.target)) {
                qualityMenu.classList.remove('show');
            }
        });

        // Initialize with original source
        document.addEventListener('DOMContentLoaded', function() {
            initPlayer(videoSources.original, 'original');
        });
    </script>
</body>
</html>